<?php
namespace App\Model\Table;

use App\Model\Entity\Unit;
use Cake\Datasource\ConnectionManager;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Units Model
 *
 * @property \Cake\ORM\Association\BelongsTo $Groups
 * @property \Cake\ORM\Association\HasMany $AreasOfOccupationProfessionals
 * @property \Cake\ORM\Association\HasMany $HealthInsurancePlans
 * @property \Cake\ORM\Association\HasMany $Hospitalizations
 * @property \Cake\ORM\Association\HasMany $IndicatorsUnits
 * @property \Cake\ORM\Association\HasMany $Origins
 * @property \Cake\ORM\Association\HasMany $Professionals
 * @property \Cake\ORM\Association\HasMany $ProfessionalsSpecialties
 * @property \Cake\ORM\Association\HasMany $Sectors
 * @property \Cake\ORM\Association\HasMany $Signs
 * @property \Cake\ORM\Association\HasMany $SurgicalComplications
 * @property \Cake\ORM\Association\HasMany $TciTypes
 * @property \Cake\ORM\Association\HasMany $Users
 * @property \Cake\ORM\Association\BelongsToMany $Patients
 * @property \Cake\ORM\Association\BelongsToMany $Users
 */
class UnitsTable extends Table
{

    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        $this->table('units');
        $this->displayField('name');
        $this->primaryKey('id');

        $this->belongsTo('Groups', [
            'foreignKey' => 'group_id',
            'joinType'   => 'INNER',
        ]);
        $this->hasMany('AreasOfOccupationProfessionals', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('HealthInsurancePlans', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('Hospitalizations', [
            'foreignKey' => 'unit_id',
        ]);
        // $this->hasMany('IndicatorsUnits', [
        //     'foreignKey' => 'unit_id',
        // ]);
        $this->hasOne('IndicatorsUnits', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('Origins', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('Professionals', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('ProfessionalsSpecialties', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('Sectors', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('Signs', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('SurgicalComplications', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('TciTypes', [
            'foreignKey' => 'unit_id',
        ]);
        $this->hasMany('Users', [
            'foreignKey' => 'unit_id',
        ]);
        $this->belongsToMany('Patients', [
            'foreignKey'       => 'unit_id',
            'targetForeignKey' => 'patient_id',
            'joinTable'        => 'patients_units',
        ]);
        $this->belongsToMany('Users', [
            'foreignKey'       => 'unit_id',
            'targetForeignKey' => 'user_id',
            'joinTable'        => 'units_users',
        ]);
        $this->hasMany('UnitPrescriptionsMetrics', [
            'foreignKey' => 'unit_id',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator)
    {
        $validator
            ->integer('id')
            ->allowEmpty('id', 'create');

        $validator
            ->requirePresence('name', 'create')
            ->notEmpty('name');

        $validator
            ->email('email')
            ->allowEmpty('email');

        $validator
            ->allowEmpty('telephone');

        $validator
            ->allowEmpty('street');

        $validator
            ->allowEmpty('city');

        $validator
            ->allowEmpty('cep');

        $validator
            ->allowEmpty('address_number');

        $validator
            ->allowEmpty('state');

        $validator
            ->allowEmpty('term_condition');

        $validator
            ->boolean('is_active')
            ->requirePresence('is_active', 'create')
            ->notEmpty('is_active');

        $validator
            ->time('tempo_de_descontaminacao_da_sala_cirurgica')
            ->allowEmpty('tempo_de_descontaminacao_da_sala_cirurgica');

        $validator
            ->time('tempo_de_descontaminacao_de_leito_da_UTI')
            ->allowEmpty('tempo_de_descontaminacao_de_leito_da_UTI');

        $validator
            ->time('tempo_de_descontaminacao_da_sala_cirurgica_contaminada')
            ->allowEmpty('tempo_de_descontaminacao_da_sala_cirurgica_contaminada');

        $validator
            ->time('tempo_de_descontaminacao_de_leito_da_unidade_de_internacao')
            ->allowEmpty('tempo_de_descontaminacao_de_leito_da_unidade_de_internacao');

        $validator
            ->allowEmpty('termo_de_consentimento_informado');

        $validator
            ->allowEmpty('modelo_de_boletim_medico_file');

        $validator
            ->allowEmpty('modelo_de_descricao_de_informacao_do_paciente_file');

        $validator
            ->allowEmpty('modelo_de_relatorio_medico_padrao_file');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules)
    {
        $rules->add($rules->isUnique(['email']));
        $rules->add($rules->existsIn(['group_id'], 'Groups'));
        return $rules;
    }

    public function addDefaultSigns($unit_id)
    {
        $data = [
            [
                'description' => 'Sinalização de medicação com baixo estoque',
                'filename'    => '6ab63175-69dd-4a49-bdcc-5a38822c64b2-baixo estoque.png',
                'unit_id'     => $unit_id,
            ],
            [
                'description' => 'Sinalização de interação medicamentosa',
                'filename'    => '3ccf5b48-e5e1-48a5-b5fc-c668c87fbdf7-interação medicamentosa.png',
                'unit_id'     => $unit_id,
            ],
            [
                'description' => 'Sinalização de medicação de alta vigilância',
                'filename'    => '8aaf6f9f-d603-40fd-a90e-24b52415fc4b-1314063744989836278red ball.png',
                'unit_id'     => $unit_id,
            ],
            [
                'description' => 'Sinalização de medicação com estoque zerado',
                'filename'    => '0871fa17-2eb9-40ea-8377-7df79e34c932-estoque zero.png',
                'unit_id'     => $unit_id,
            ],
            [
                'description' => 'Sinalização de medicação não padronizada',
                'filename'    => 'fcc22837-73ad-4693-a3ae-c4e9c156f4d9-baixo nao padronizado.png',
                'unit_id'     => $unit_id,
            ],
            [
                'description' => 'Medicação regular (sem restrição e padronizada)',
                'filename'    => 'cad5d785-626e-4f2b-9239-581d00caaa70-medicação regular.png',
                'unit_id'     => $unit_id,
            ],
        ];

        $conn = ConnectionManager::get('default');

        foreach ($data as $key => $value) {
            $conn->execute('INSERT INTO signs (description, filename, unit_id) VALUES (:description, :filename, :unit_id);',
                [
                    'description' => $value['description'],
                    'filename'    => $value['filename'],
                    'unit_id'     => $value['unit_id'],
                ]);
        }

    }
}
