<?php
namespace App\Model\Table;

use App\Model\Entity\Bundle;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use MedidasDispersao\MedidasDispersao;

require_once ROOT . DS . "vendor" . DS . "Zurubabel" . DS . "MedidasDispersao.php";

/**
 * Bundles Model
 *
 * @property \Cake\ORM\Association\BelongsTo $Professionals
 * @property \Cake\ORM\Association\BelongsTo $Hospitalizations
 * @property \Cake\ORM\Association\BelongsTo $Screenings
 */
class BundlesTable extends Table
{

    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config)
    {
        parent::initialize($config);

        $this->table('bundles');
        $this->displayField('id');
        $this->primaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Professionals', [
            'foreignKey' => 'professional_id',
            'joinType' => 'INNER',
        ]);
        $this->belongsTo('Hospitalizations', [
            'foreignKey' => 'hospitalization_id',
        ]);
        $this->belongsTo('Screenings', [
            'foreignKey' => 'screening_id',
        ]);
        $this->hasMany('DailyBundles', [
            'foreignKey' => 'bundle_id',
        ]);
        $this->belongsToMany('ImmediateComplications', [
            'foreignKey' => 'bundle_id',
            'targetForeignKey' => 'immediate_complication_id',
            'joinTable' => 'bundles_immediate_complications',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator)
    {
        $validator
            ->integer('id')
            ->allowEmpty('id', 'create');

        $validator
        // ->date('date_instalation')
        ->requirePresence('date_instalation', 'create')
            ->notEmpty('date_instalation');

        $validator
        // ->time('hour_instalation')
        ->requirePresence('hour_instalation', 'create')
            ->notEmpty('hour_instalation');

        $validator
            ->requirePresence('type_of_device', 'create')
            ->notEmpty('type_of_device');

        $validator
            ->boolean('emergency')
            ->allowEmpty('emergency');

        $validator
            ->boolean('is_risk_of_bleeding_verified')
            ->allowEmpty('is_risk_of_bleeding_verified');

        $validator
            ->boolean('is_the_procedure_prescribed')
            ->allowEmpty('is_the_procedure_prescribed');

        $validator
            ->allowEmpty('device_observation');

        $validator
            ->boolean('proper_handwashing')
            ->allowEmpty('proper_handwashing');

        $validator
            ->boolean('use_of_paramentation')
            ->allowEmpty('use_of_paramentation');

        $validator
            ->boolean('use_of_antisepsis')
            ->allowEmpty('use_of_antisepsis');

        $validator
            ->boolean('use_of_field_with_maxium_barrier')
            ->allowEmpty('use_of_field_with_maxium_barrier');

        $validator
            ->boolean('use_of_sterile_gel')
            ->allowEmpty('use_of_sterile_gel');

        $validator
            ->boolean('suitable_catheter_progression')
            ->allowEmpty('suitable_catheter_progression');

        $validator
            ->boolean('urine_output_cauterization')
            ->allowEmpty('urine_output_cauterization');

        $validator
            ->boolean('adequate_catheter_fixation')
            ->allowEmpty('adequate_catheter_fixation');

        $validator
            ->boolean('confirmation_of_position_with_us')
            ->allowEmpty('confirmation_of_position_with_us');

        $validator
            ->integer('checklist_observation')
            ->allowEmpty('checklist_observation');

        $validator
            ->integer('number_of_retries')
            ->allowEmpty('number_of_retries');

        $validator
            ->boolean('single_procedure')
            ->allowEmpty('single_procedure');

        $validator
            ->integer('number_of_professionals')
            ->allowEmpty('number_of_professionals');

        $validator
            ->boolean('blood_on_catheter')
            ->allowEmpty('blood_on_catheter');

        $validator
            ->boolean('successful_integration')
            ->allowEmpty('successful_integration');

        $validator
        // ->date('date_of_withdrawal')
        ->allowEmpty('date_of_withdrawal');

        $validator
            ->date('withdrawal_time')
            ->allowEmpty('withdrawal_time');

        $validator
            ->boolean('accidental_removal')
            ->allowEmpty('accidental_removal');

        $validator
            ->allowEmpty('reason_for_inclusion');

        $validator
            ->time('reinsert_time')
            ->allowEmpty('reinsert_time');

        $validator
        // ->date('reinsert_date')
        ->allowEmpty('reinsert_date');

        $validator
            ->boolean('citostimia')
            ->allowEmpty('citostimia');

        return $validator;
    }

    public function getRelatorioData($hospitalizations, $date_before, $date_after)
    {
        $calc_obj = new MedidasDispersao();

        $data =
            [
            'total_de_bundles' => 0,
            'total_internacoes' => 0,

            //total de paciente que fez o bundle vascular
            'arr_bundle_vascular_por_pacientes' => [],
            'tot_bundle_vascular_por_pacientes' => 0,

            //Pacientes com bundle de cateter vascular / Pacientes com cateter arterial
            'arr_internacao_vascular_por_arterial' => [],
            'tot_internacao_vascular_por_arterial' => 0,

            'arr_internacao_vascular_por_venoso' => [],
            'tot_internacao_vascular_por_venoso' => 0,

            'arr_internacao_vascular_por_hemodialise' => [],
            'tot_internacao_vascular_por_hemodialise' => 0,

            'arr_internacao_vascular_por_picc' => [],
            'tot_internacao_vascular_por_picc' => 0,

            //Cateter venoso no período em relação ao total de bundles
            'tot_cateter_venoso_por_bundle' => 0,
            'tot_cateter_arterial_por_bundle' => 0,

            'percen_pacientes_bundle' => 0,

            //Procedimentos realizados com picc
            'tot_procedimentos_utilizando_picc' => 0,
            'percen_procedimentos_utilizando_picc' => 0,

            //Procedimentos realizados com hemodiálise
            'tot_procedimentos_utilizando_hemodialise' => 0,
            'percen_procedimentos_utilizando_hemodialise' => 0,

            //Local de inserção
            'local_insercao' => [
                'tot_arteria_axilar' => 0,
                'percen_arteria_axilar' => 0,

                'tot_arteria_braquial' => 0,
                'percen_arteria_braquial' => 0,

                'tot_arteria_femoral' => 0,
                'percen_arteria_femoral' => 0,

                'tot_arteria_pediosa' => 0,
                'percen_arteria_pediosa' => 0,

                'tot_arteria_radial' => 0,
                'percen_arteria_radial' => 0,

                'tot_arteria_ulnar' => 0,
                'percen_arteria_ulnar' => 0,

                'tot_arteria_umeral' => 0,
                'percen_arteria_umeral' => 0,

                'tot_veia_jugular_inter' => 0,
                'percen_veia_jugular_inter' => 0,

                'tot_veia_jugular_exter' => 0,
                'percen_veia_jugular_exter' => 0,

                'tot_veia_radial' => 0,
                'percen_veia_radial' => 0,

                'tot_veia_subclavia' => 0,
                'percen_veia_subclavia' => 0,

                'tot_veia_basilica' => 0,
                'percen_veia_basilica' => 0,

                'tot_veia_pediosa' => 0,
                'percen_veia_pediosa' => 0,

                'tot_veia_cefalica' => 0,
                'percen_veia_cefalica' => 0,

                'tot_veia_axilar' => 0,
                'percen_veia_axilar' => 0,

                'tot_veia_femoral' => 0,
                'percen_veia_femoral' => 0,

                'tot_veia_mediana' => 0,
                'percen_veia_mediana' => 0,

                'tot_veia_subclavia_infra' => 0,
                'percen_veia_subclavia_infra' => 0,

                'tot_veia_subclavia_supra' => 0,
                'percen_veia_subclavia_supra' => 0,

                'tot_veia_temporal' => 0,
                'percen_veia_temporal' => 0,

                'tot_veia_umbilical' => 0,
                'percen_veia_umbilical' => 0,

                'tot_outra' => 0,
                'percen_outra' => 0,

            ],
            //Procedimentos com uma única tentativa
            'tot_tentativa_unica' => 0,
            'percen_tentativa_unica' => 0,

            //Houve complicação imediata
            'tot_houve_complicacao_imediata' => 0,
            'percen_houve_complicacao_imediata' => 0,
            'tot_qtd_complicacao_imediata' => 0,
            'percen_qtd_complicacao_imediata' => 0,

            //vesical
            //TOTAL DE BUNDLES DE PASSAGEM DE CATETER VESICAL DE DEMORA
            'tot_bundle_vesical_demora' => 0,

            //VALOR PERCENTUAL DE BUNDLE DE CATETER VESICAL DE DEMORA / PACIENTE
            'percen_bundle_vesical_paciente' => 0,

            //total de pacientes com bundle de cateter vesical de demora
            'arr_pacient_bundle_vesical' => [],
            'tot_pacient_bundle_vesical' => 0,

            //total de medidas de suporte com cateter vesical
            'tot_procedimento_cateter_vesical' => 0,

            //total de pacientes(internacoes) com procedimento de cateter vesical
            'arr_internacao_procedimento_vesical' => [],
            'tot_internacao_procedimento_vesical' => 0,

            //VALOR TOTAL DE PACIENTES(internacoes) COM BUNDLE DE CATETER VESICAL DE DEMORA / VALOR TOTAL DE PACIENTES COM CATETER VESICAL MARCADO EM PROCEDIMENTOS
            'percen_pacient_bundle_vesical_procedimento' => 0,

            //VALOR ABSOLUTO E PERCENTUAL DE DISPOSITIVOS INSERIDOS NO PERÍODO EM RELAÇÃO AO TOTAL DE BUNDLES
            'tot_dispositivos_inseridos' => 0,
            'percen_dispositivos_inseridos' => 0,

        ];
        $percentual = [];

        foreach ($hospitalizations as $hospitalization)
        {
            $data['total_internacoes']++;

            foreach ($hospitalization["bundles"] as $key_1 => $bundle)
            {
                $bundle_arr = (array) $bundle;
                $i = 0;

                if ($bundle['date_instalation'] >= $date_before &&
                    $bundle['date_instalation'] <= $date_after
                )
                {
                    $data['total_de_bundles']++;

                    // calc
                    if (count($bundle['immediate_complications']) > 0
                    )
                    {
                        $data['tot_houve_complicacao_imediata']++;
                        $data['tot_qtd_complicacao_imediata'] = count($bundle['immediate_complications']);
                    }

                    //calc numero de procedimentos com tentaiva unica
                    if ($bundle['number_of_retries'] == 1
                    )
                    {
                        $data['tot_tentativa_unica']++;
                    }

                    //Calc o local de inserção
                    switch ($bundle['local_insercao'])
                    {
                        case 0:
                            $data['local_insercao']['tot_arteria_axilar']++;
                            break;
                        case '1':
                            $data['local_insercao']['tot_arteria_braquial']++;
                            break;
                        case '2':
                            $data['local_insercao']['tot_arteria_femoral']++;
                            break;
                        case '3':
                            $data['local_insercao']['tot_arteria_pediosa']++;
                            break;
                        case '4':
                            $data['local_insercao']['tot_arteria_radial']++;
                            break;
                        case '5':
                            $data['local_insercao']['tot_arteria_ulnar']++;
                            break;
                        case '6':
                            $data['local_insercao']['tot_arteria_umeral']++;
                            break;
                        case '7':
                            $data['local_insercao']['tot_veia_jugular_inter']++;
                            break;
                        case '8':
                            $data['local_insercao']['tot_veia_jugular_exter']++;
                            break;
                        case '9':
                            $data['local_insercao']['tot_veia_radial']++;
                            break;
                        case '10':
                            $data['local_insercao']['tot_veia_subclavia']++;
                            break;
                        case '11':
                            $data['local_insercao']['tot_veia_basilica']++;
                            break;
                        case '12':
                            $data['local_insercao']['tot_veia_pediosa']++;
                            break;
                        case '13':
                            $data['local_insercao']['tot_veia_cefalica']++;
                            break;
                        case '14':
                            $data['local_insercao']['tot_veia_axilar']++;
                            break;
                        case '15':
                            $data['local_insercao']['tot_veia_femoral']++;
                            break;
                        case '16':
                            $data['local_insercao']['tot_veia_mediana']++;
                            break;
                        case '17':
                            $data['local_insercao']['tot_veia_subclavia_infra']++;
                            break;
                        case '18':
                            $data['local_insercao']['tot_veia_subclavia_supra']++;
                            break;
                        case '19':
                            $data['local_insercao']['tot_veia_temporal']++;
                            break;
                        case '20':
                            $data['local_insercao']['tot_veia_umbilical']++;
                            break;
                        case '21':
                            $data['local_insercao']['tot_outra']++;
                            break;
                    }

                    //calcula total de paciente que fez o bundle vascular
                    if (in_array($hospitalization->id, $data['arr_bundle_vascular_por_pacientes']) == false)
                    {
                        array_push($data['arr_bundle_vascular_por_pacientes'], $hospitalization->id);
                        $data['tot_bundle_vascular_por_pacientes']++;
                    }

                    if (stripos($bundle['type_of_device'], 'arteria') !== false)
                    {
                        if (in_array($hospitalization->id, $data['arr_internacao_vascular_por_arterial']) == false)
                        {
                            array_push($data['arr_internacao_vascular_por_arterial'], $hospitalization->id);
                        }
                    }

                    if (stripos($bundle['type_of_device'], 'venoso') !== false)
                    {
                        if (in_array($hospitalization->id, $data['arr_internacao_vascular_por_venoso']) == false)
                        {
                            array_push($data['arr_internacao_vascular_por_venoso'], $hospitalization->id);
                        }
                    }

                    if (stripos($bundle['type_of_device'], 'hemodialise') !== false)
                    {
                        $data['tot_procedimentos_utilizando_hemodialise']++;
                        if (in_array($hospitalization->id, $data['arr_internacao_vascular_por_hemodialise']) == false)
                        {
                            array_push($data['arr_internacao_vascular_por_hemodialise'], $hospitalization->id);
                        }
                    }

                    if (stripos($bundle['type_of_device'], 'picc') !== false)
                    {
                        $data['tot_procedimentos_utilizando_picc']++;

                        if (in_array($hospitalization->id, $data['arr_internacao_vascular_por_picc']) == false)
                        {
                            array_push($data['arr_internacao_vascular_por_picc'], $hospitalization->id);
                        }
                    }

                    if (stripos($bundle['type_of_device'], 'vesical de demora') !== false)
                    {
                        $data['tot_bundle_vesical_demora']++;

                        if (in_array($hospitalization->id, $data['arr_pacient_bundle_vesical']) == false)
                        {
                            array_push($data['arr_pacient_bundle_vesical'], $hospitalization->id);
                            $data['tot_pacient_bundle_vesical']++;
                        }
                    }

                    foreach ($bundle_arr as $key_2 => $value_2)
                    {
                        $i++;
                        if ($i == 2) //acessa as propriades do obj
                        {
                            foreach ($value_2 as $key_3 => $value_3)
                            {
                                //$data[$key_3] = $value_3;
                                formatArr($value_3, $data, $key_3);
                            }
                        }
                    }
                }
            }

            if (isset($hospitalization["supportive_measures"]))
            {
                foreach ($hospitalization["supportive_measures"] as $key_1 => $supportive_measure)
                {
                    if ($supportive_measure['data_inicio'] >= $date_before &&
                        $supportive_measure['data_termino'] <= $date_after
                    )
                    {
                        $data["tot_dispositivos_inseridos"]++;
                        if (stripos($supportive_measure['type'], 'cateter vesical') !== false)
                        {
                            $data['tot_procedimento_cateter_vesical']++;

                            if (in_array($hospitalization->id, $data['arr_internacao_procedimento_vesical']) == false)
                            {
                                array_push($data['arr_internacao_procedimento_vesical'], $hospitalization->id);
                                $data['tot_internacao_procedimento_vesical']++;
                            }
                        }
                    }
                }
            }
        }

        foreach ($data as $key => $value)
        {
            $marcados = 0;

            if (isset($data[$key]['percentual']) && isset($data[$key]['arr']))
            {
                $len = count($data[$key]['arr']);

                for ($i = 0; $i < $len; $i++)
                {
                    if ($data[$key]['arr'][$i] || $data[$key]['arr'][$i] !== 0)
                    {
                        $marcados++;
                    }
                }

                $data[$key]['total'] = count($data[$key]['arr']);
                $data[$key]['soma'] = 0;
                $data[$key]['percentual'] = $calc_obj->calculate_percentage($marcados, $data['total_de_bundles']);
                $data[$key]['media'] = $calc_obj->calc_average($data[$key]['arr']);
                $data[$key]['mediana'] = $calc_obj->calculate_median($data[$key]['arr']);
                $data[$key]['desvio_padrao'] = $calc_obj->standard_deviation($data[$key]['arr']);

                //calcula o total
                if (count($data[$key]['arr']) > 0)
                {
                    foreach ($data[$key]['arr'] as $attr)
                    {
                        if (is_numeric($attr))
                        {
                            $data[$key]['soma'] += $attr;
                        }
                    }
                }
            }
        }

        //calc Bundles de cateter vascular por paciente
        if ($data['total_internacoes'] > 0 && $data['total_de_bundles'] > 0)
        {
            $data['bundle_vascular_por_pacientes'] = ($data['total_de_bundles'] / $data['total_internacoes']) * 100;
        }

        //calc pacientes bom bundle vascular por arterial
        if ($data['tot_bundle_vascular_por_pacientes'] > 0 &&
            count($data['arr_internacao_vascular_por_arterial']) > 0)
        {
            $tot_paciente_cateter_arterial = count($data['arr_internacao_vascular_por_arterial']);
            $data['tot_internacao_vascular_por_arterial'] = ($tot_paciente_cateter_arterial * 100) / $data['tot_bundle_vascular_por_pacientes'];
            $data['tot_internacao_vascular_por_arterial'] = round($data['tot_internacao_vascular_por_arterial'], 2);

            $data['tot_cateter_arterial_por_bundle'] = round(($tot_paciente_cateter_arterial * 100) / $data['total_de_bundles'], 2);
        }

        //calc pacientes bom bundle vascular por venoso
        if ($data['tot_bundle_vascular_por_pacientes'] > 0 &&
            count($data['arr_internacao_vascular_por_venoso']) > 0)
        {
            $tot_paciente_cateter_venoso = count($data['arr_internacao_vascular_por_venoso']);
            $data['tot_internacao_vascular_por_venoso'] = ($tot_paciente_cateter_venoso * 100) / $data['tot_bundle_vascular_por_pacientes'];
            $data['tot_internacao_vascular_por_venoso'] = round($data['tot_internacao_vascular_por_venoso'], 2);

            $data['tot_cateter_venoso_por_bundle'] = round(($tot_paciente_cateter_venoso * 100) / $data['total_de_bundles'], 2);
        }

        //calc pacientes bom bundle vascular por hemodialise
        if ($data['tot_bundle_vascular_por_pacientes'] > 0 &&
            count($data['arr_internacao_vascular_por_hemodialise']) > 0)
        {
            $tot_paciente_cateter_hemodialise = count($data['arr_internacao_vascular_por_hemodialise']);
            $data['tot_internacao_vascular_por_hemodialise'] = ($tot_paciente_cateter_hemodialise * 100) / $data['tot_bundle_vascular_por_pacientes'];
            $data['tot_internacao_vascular_por_hemodialise'] = round($data['tot_internacao_vascular_por_hemodialise'], 2);
        }

        //calc pacientes bom bundle vascular por picc
        if ($data['tot_bundle_vascular_por_pacientes'] > 0 &&
            count($data['arr_internacao_vascular_por_picc']) > 0)
        {
            $tot_paciente_cateter_picc = count($data['arr_internacao_vascular_por_picc']);
            $data['tot_internacao_vascular_por_picc'] = ($tot_paciente_cateter_picc * 100) / $data['tot_bundle_vascular_por_pacientes'];
            $data['tot_internacao_vascular_por_picc'] = round($data['tot_internacao_vascular_por_picc'], 2);
        }

        if ($data['total_de_bundles'] > 0 &&
            $data['tot_procedimentos_utilizando_picc'] > 0)
        {
            $data['percen_procedimentos_utilizando_picc'] = ($data['tot_procedimentos_utilizando_picc'] * 100) / $data['total_de_bundles'];
            $data['percen_procedimentos_utilizando_picc'] = round($data['percen_procedimentos_utilizando_picc'], 2);
        }

        if ($data['total_de_bundles'] > 0 &&
            $data['tot_procedimentos_utilizando_hemodialise'] > 0)
        {
            $data['percen_procedimentos_utilizando_hemodialise'] = ($data['tot_procedimentos_utilizando_hemodialise'] * 100) / $data['total_de_bundles'];
            $data['percen_procedimentos_utilizando_hemodialise'] = round($data['percen_procedimentos_utilizando_hemodialise'], 2);
        }

        if ($data['total_de_bundles'] > 0 &&
            $data['tot_tentativa_unica'] > 0)
        {
            $data['percen_tentativa_unica'] = ($data['tot_tentativa_unica'] * 100) / $data['total_de_bundles'];
            $data['percen_tentativa_unica'] = round($data['percen_tentativa_unica'], 2);
        }

        //calc local de insercao
        foreach ($data['local_insercao'] as $local_key => $local_value)
        {
            if (stripos($local_key, 'tot_') !== false)
            {
                if ($local_value > 0)
                {
                    $percent_key = str_replace("tot_", "percen_", $local_key);
                    $data['local_insercao'][$percent_key] = ($local_value * 100) / $data['total_de_bundles'];
                    $data['local_insercao'][$percent_key] = round($data['local_insercao'][$percent_key], 2);
                }
            }
        }

        //calc complicacoes imediatas
        if ($data['total_de_bundles'] > 0 &&
            $data['tot_houve_complicacao_imediata'] > 0)
        {
            $data['percen_houve_complicacao_imediata'] = ($data['tot_houve_complicacao_imediata'] * 100) / $data['total_de_bundles'];
            $data['percen_houve_complicacao_imediata'] = round($data['percen_houve_complicacao_imediata'], 2);
        }

        //calc bundle vesical de demora por paciente
        if ($data['total_internacoes'] > 0 &&
            $data['tot_bundle_vesical_demora'] > 0)
        {
            $data['percen_bundle_vesical_paciente'] = ($data['tot_bundle_vesical_demora'] * 100) / $data['total_internacoes'];
            $data['percen_bundle_vesical_paciente'] = round($data['percen_bundle_vesical_paciente'], 2);
        }

        //calc VALOR TOTAL DE PACIENTES(internacoes) COM BUNDLE DE CATETER VESICAL DE DEMORA / VALOR TOTAL DE PACIENTES COM CATETER VESICAL MARCADO EM PROCEDIMENTOS
        if ($data['tot_pacient_bundle_vesical'] > 0 &&
            $data['tot_internacao_procedimento_vesical'] > 0)
        {
            $data['percen_pacient_bundle_vesical_procedimento'] = ($data['tot_pacient_bundle_vesical']) / $data['tot_internacao_procedimento_vesical'];
            $data['percen_pacient_bundle_vesical_procedimento'] = round($data['percen_bundle_vesical_paciente'], 2);
        }

        //calc VALOR ABSOLUTO E PERCENTUAL DE DISPOSITIVOS INSERIDOS NO PERÍODO EM RELAÇÃO AO TOTAL DE BUNDLES
        if ($data['total_de_bundles'] > 0 &&
            $data['tot_dispositivos_inseridos'] > 0)
        {
            $data['percen_dispositivos_inseridos'] = ($data['tot_dispositivos_inseridos'] * 100) / $data['total_de_bundles'];
            $data['percen_dispositivos_inseridos'] = round($data['percen_dispositivos_inseridos'], 2);
        }

        return $data;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules)
    {
        $rules->add($rules->existsIn(['professional_id'], 'Professionals'));
        $rules->add($rules->existsIn(['hospitalization_id'], 'Hospitalizations'));
        //$rules->add($rules->existsIn(['screening_id'], 'Screenings'));
        return $rules;
    }
}
