<?php
namespace App\Controller;

use App\Controller\AppController;
use Cake\ORM\TableRegistry;
use Cake\I18n\Time;
use Cake\Database\Expression\QueryExpression;

/**
 * Prescriptions Controller
 *
 * @property \App\Model\Table\PrescriptionsTable $Prescriptions
 */
class PrescriptionsController extends AppController
{
    public function initialize()
    {
        parent::initialize();
        $this->loadComponent('Auth', [
            'authorize' => 'Controller',
        ]);
        $this->loadComponent('Permission');

        // View or Controller
        $this->set('title', 'Prescrições');

        $session = $this->request->session();

        $pacient_id = $session->read('selected-pacient');
        $this->set('pacient_id', $pacient_id);

        $unit_id = $session->read('selected-unit');
        $this->set('unit_id', $unit_id);

        $hospitalization_id = $session->read('selected-hospitalization');
        $this->set('hospitalization_id', $hospitalization_id);

        $current_state = $this->request->session()->read('current-state');
        $this->set('current_state', $current_state);
    }

    public function isAuthorized($user)
    {
        $action = $this->request->params['action'];
        $privilege = $this->request->session()->read('selected-unit-privilege');
        return $this->Permission->hasPermission($privilege, 'Prescriptions', $action);
        //return parent::isAuthorized($user);
    }

    /**
     * Index method
     *
     * @return \Cake\Network\Response|null
     */
    public function index()
    {
        $session = $this->request->session();
        $hospitalization_id = $session->read('selected-hospitalization');
        $screening_id = $this->request->session()->read('screening_id');
        $current_state = $this->request->session()->read('current-state');
        $lastPrescriptionId = 0;

        if ($current_state == 'internacao') {
            $allPresc = $this->Prescriptions->find(
                'all',
                [
                    'conditions' => ['Prescriptions.hospitalization_id' => $hospitalization_id],
                    'order' => ['id' => 'DESC'],
                    'limit' => 1,
                ]
            );

            $lastPrescription = $allPresc->toArray();

            if (isset($lastPrescription[0]['id'])) {
                $lastPrescriptionId = $lastPrescription[0]['id'];
            }

            $this->paginate = [
                'contain' => [],
                'conditions' => [
                    'Prescriptions.is_active' => 1,
                    'Prescriptions.hospitalization_id' => $hospitalization_id,
                ],
                'order' => ['created' => 'desc'],
            ];
        } else {
            $allPresc = $this->Prescriptions->find(
                'all',
                [
                    'conditions' => ['Prescriptions.screening_id' => $screening_id],
                    'order' => ['id' => 'DESC'],
                    'limit' => 1,
                ]
            );

            $lastPrescription = $allPresc->toArray();

            if (isset($lastPrescription[0]['id'])) {
                $lastPrescriptionId = $lastPrescription[0]['id'];
            }

            $this->paginate = [
                'contain' => [],
                'conditions' => [
                    'Prescriptions.is_active' => 1,
                    'Prescriptions.screening_id' => $screening_id,
                ],
                'order' => ['created' => 'desc'],
            ];
        }

        $prescriptions = $this->paginate($this->Prescriptions);

        $this->set(compact('prescriptions', 'lastPrescriptionId', 'lastPrescription', 'screening_id'));
        $this->set('_serialize', ['prescriptions']);
    }


    public function prescriptionsDashboard()
    {

        $unitId               = $this->request->session()->read('selected-unit');
        $sectorId             = $this->request->query('sector_id');
        $sectorsTable         = TableRegistry::get('Sectors');

        // unit list of sectors
        $unitAvaiableSectors = $sectorsTable->find('list', [
          'keyField' => 'id',
          'valueField' => 'name'
        ])
        ->where(["Sectors.unit_id" => $unitId]);

        if (!empty($sectorId)) {
            $metrics = $this->getSectorPrescriptionsMetrics($sectorId);
        }
        else {
           $metrics = $this->getUnitPrescriptionsMetrics($unitId);
        }

        $this->set(compact('metrics', 'unitAvaiableSectors'));

    }


    private function getSectorPrescriptionsMetrics($sectorId=null) {
      $baseModel            = "Sectors";
      $sectorPrescMetrTable   = TableRegistry::get('SectorPrescriptionsMetrics');

      $query = $sectorPrescMetrTable->find('all')
      ->where(['SectorPrescriptionsMetrics.sector_id' => $sectorId]);

      $lastSectorMetrics = (!$query->isEmpty()) ? $query->first()->toArray() : [];

      $currentSectorMetrics = $this->getCurrentMetrics([$sectorId], $baseModel);

      return $this->compareMetrics($lastSectorMetrics, $currentSectorMetrics, $baseModel);
    }


    private function getUnitPrescriptionsMetrics($unitId=null) {
      $baseModel            = "Units";
      $sectorsTable         = TableRegistry::get('Sectors');
      $unitPrescMetrTable   = TableRegistry::get('UnitPrescriptionsMetrics');

      $unitSectorsIds = $sectorsTable->find('list')
      ->select(['id'])
      ->where(["Sectors.unit_id" => $unitId]);

      $query = $unitPrescMetrTable->find('all')
      ->where(['UnitPrescriptionsMetrics.unit_id'=>$unitId]);


      $lastUnitMetrics = (!$query->isEmpty()) ? $query->first()->toArray() : [];

      $currentUnitMetrics = $this->getCurrentMetrics($unitSectorsIds, $baseModel);

      return $this->compareMetrics($lastUnitMetrics, $currentUnitMetrics, $baseModel);
    }


    private function getCurrentMetrics($ids, $baseModel) {

         $hospitalizationsIds = [];
         $results = [];
         switch ($baseModel) {

            case 'Sectors':
                  $hospitalizationsIds = $this->Prescriptions->Hospitalizations->find('list')
                  ->select(['id'])
                  ->where(["Hospitalizations.sector_id" => $ids[0] ]);

                  // assinging the sector id to the results
                  $results['sector_id'] = $ids[0];

               break;

            default:
                  $hospitalizationsIds = $this->Prescriptions->Hospitalizations->find('list')
                  ->select(['id'])
                  ->where(["Hospitalizations.sector_id IN" => $ids]);
               break;
         }


         //counting items
         $systemMedicationsTable = TableRegistry::get('SystemMedications');
         $totalItems             = $systemMedicationsTable->find('all')->count();


          //counting active prescriptions
          $totalActivePrescriptions = $this->Prescriptions->find('all')
          ->where(['Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->where(['Prescriptions.is_active' => true])
          ->count();

          //counting validated prescriptions
          $totalValidatedPrescriptions =  $this->Prescriptions->find('all')
          ->where(['Prescriptions.validated' => true,'Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->count();

          //counting surveillance signed meds
          $totalHighSurveillanceItems = $this->Prescriptions->find('all')
          ->where(['Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->matching('PrescriptionMedications', function ($q) {
            return $q->where(['PrescriptionMedications.surveillance_signaling' => 'Alta Vigilância']);
          })
          ->count();

          //counting potential dangerous meds
          $totalPotentiallyDangerousItems = $this->Prescriptions->find('all')
          ->where(['Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->matching('PrescriptionMedications', function ($q) {
             return $q->where(['PrescriptionMedications.surveillance_signaling' => 'Med. Potencialmente Perigosas']);
          })
          ->count();

          // counting prescription with low stock medications
          $totalPrescriptionWithMedsLowStock = $this->Prescriptions->find('all')
          ->where(['Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->matching('PrescriptionMedications', function ($q) {
              return $q->where(['PrescriptionMedications.quantity <' => 5]);
          })
          ->count();

         // counting prescription with  medications with 0 number in stock
          $totalPrescriptionWithMedsZeroStock = $this->Prescriptions->find('all')
          ->where(['Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->matching('PrescriptionMedications', function ($q) {
              return $q->where(['PrescriptionMedications.quantity ' => 0]);
          })
          ->count();

          // counting prescription with standardized medications
          $totalPrescriptionWithStandardizedMeds = $this->Prescriptions->find('all')
          ->where(['Prescriptions.hospitalization_id IN ' => $hospitalizationsIds])
          ->contain(['PrescriptionMedications'])
          ->matching('PrescriptionMedications', function ($q) {
              return $q->where(['PrescriptionMedications.surveillance_signaling <>' =>  "Não Padronizada"]);
          })
          ->count();

          $results['total_items']                          = $totalItems;
          $results['total_high_surveillance_items']        = $totalHighSurveillanceItems;
          $results['total_potentially_dangerous_items']    = $totalPotentiallyDangerousItems;
          $results['total_active_prescriptions']           = $totalActivePrescriptions;
          $results['prescription_with_meds_low_stock']     = $totalPrescriptionWithMedsLowStock;
          $results['prescription_with_meds_zero_stock']    = $totalPrescriptionWithMedsZeroStock;
          $results['total_validated_prescriptions']        = $totalValidatedPrescriptions;
          $results['prescription_with_standardized_meds']  = $totalPrescriptionWithStandardizedMeds;

          return $results;
    }

    private function compareMetrics($last, $current, $baseModel) {

      $hasChanged = false;
      $metricsToRender = [];
      $skipFields = [
          'id',
          'created',
          'modified',
          'unit_id',
          'sector_id'
      ];


      if (!empty($last)) {

        foreach ($last as $key => $value) {

            if(!in_array($key, $skipFields)) {

              // flag to save new metric
              if ($current[$key] != $value) { $hasChanged = true; }

              // modification status
              if ($current[$key] > $value) {
                $status = "up";
              }
              elseif ($current[$key] < $value) {
                $status = "down";
              }
              else {
                $status = "nochange";
              }

              $metricsToRender[$key] = [
                "value"   => $current[$key],
                "status"  => $status
              ];

            }
        }

      }
      else {

        // if we don't have a last metric saved in the database
        $hasChanged = true;
        foreach ($current as $key => $value) {
            $metricsToRender[$key] = [
                  "value"   => $value,
                  "status"  => "nochange"
            ];
        }

      }

      if ($hasChanged) {

        if ($baseModel == "Units") {

          $table = TableRegistry::get("UnitPrescriptionsMetrics");
          $entity = $table->newEntity();
          $entity->unit_id = $this->request->session()->read('selected-unit');

        }
        elseif ($baseModel == "Sectors") {

          $table = TableRegistry::get("SectorPrescriptionsMetrics");
          $entity = $table->newEntity();

        }

        $entity = $table->patchEntity($entity, $current);
        $table->save($entity);

      }

      return $metricsToRender;
    }

    public function pharmaceuticalValidation()
    {
        $session = $this->request->session();
        $hospitalization_id = $session->read('selected-hospitalization');
        $screening_id = $this->request->session()->read('screening_id');
        $current_state = $this->request->session()->read('current-state');
        $lastPrescriptionId = 0;

        $query = $this->Prescriptions->find('all')
          ->contain(['Sectors'])
          ->where([
            'Prescriptions.hospitalization_id' => $hospitalization_id,
            'Prescriptions.is_active' => 1
          ]);

        $searchParam = (!empty($this->request->query('q'))) ? $this->request->query('q') : '';
        $searchType  = (!empty($this->request->query('q_type'))) ? $this->request->query('q_type') : '';

        if (!empty($searchType) && !empty($searchParam)) {

          switch ($searchType) {

            case 'name':

                  $query = $query
                    ->where(['Prescriptions.name LIKE' => "%".$searchParam."%"]);

              break;

            case 'date':

                $searchParam = \DateTime::createFromFormat('d/m/Y', $searchParam)->format('Y-m-d');
                $query = $query
                  ->where(['Prescriptions.created LIKE' => "%".$searchParam."%"]);

              break;

            case 'sector':

                  $query = $query
                    ->contain(['Sectors'])
                    ->matching('Sectors', function ($q) use($searchParam) {
                      return $q->where(['Sectors.name LIKE' => "%".$searchParam."%"]);
                    });

              break;

            default:

              break;
          }
        }

        $prescriptions = $this->paginate($query);
        $this->set(compact('prescriptions', 'screening_id'));
        $this->set('_serialize', ['prescriptions']);
    }

    public function pharmaceuticalItemsValidation($id = null)
    {
        $prescription = $this->Prescriptions->get($id, [
            'contain' => [
               'PrescriptionMedications'=> function ($q) {
                   return $q
                    ->contain(['SystemMedications'])
                    ->where(['PrescriptionMedications.is_active' => true]);
               },
               'Professionals.Users'
            ]
        ]);

        $patient_table           = TableRegistry::get('Patients');
        $hospitalizations_table  = TableRegistry::get('Hospitalizations');
        $unitsTable              = TableRegistry::get('Units');

        $patient = $patient_table->get($this->request->session()->read('selected-pacient'));
        $unit    = $unitsTable->get($this->request->session()->read('selected-unit'));

        $hospitalization = $hospitalizations_table->get($this->request->session()->read('selected-hospitalization'),[
           'contain'=>['HistoricPatientMovements.Rooms', 'Sectors']
        ]);


        if ($this->request->is(['patch', 'post', 'put'])) {
            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data);

            if ($this->Prescriptions->save($prescription)) {
                $this->Flash->success(__('Medicações validadas com sucesso.'));
                return $this->redirect(['action' => 'pharmaceutical-validation']);
            } else {
                $this->Flash->error(__('Não foi possível validar os itens, por favor tente novamente.'));
            }
        }



        $this->set(compact('prescription', 'patient', 'hospitalization', 'unit'));
        $this->set('_serialize', ['prescription']);
    }

    public function prescriptionCopy($prescription_id = null)
    {
        $session = $this->request->session();
        $hospitalization_id = $session->read('selected-hospitalization');
        $screening_id = $this->request->session()->read('screening_id');
        $current_state = $this->request->session()->read('current-state');

        // search logics
        $searchParam = (!empty($this->request->query('q'))) ? $this->request->query('q') : '';
        $dateCondition = [];

        if (!empty($searchParam)) {
          $dateCondition = ['PrescriptionMedications.created LIKE' => '%'.$searchParam.'%'];
        }

        // date ranges init and format
        $today = Time::now();
        $fiveDaysAgo = new Time('5 days ago');

        $today       = $today->i18nFormat('yyyy-MM-dd');
        $fiveDaysAgo = $fiveDaysAgo->i18nFormat('yyyy-MM-dd');

        // query to load itens inside the table
        $prescriptionMedications = $this->Prescriptions->PrescriptionMedications->find('all')
        ->contain(['Prescriptions','SystemMedications'])
        ->where($dateCondition)
        ->andWhere(function($exp) use($fiveDaysAgo, $today) {
            return $exp->between('PrescriptionMedications.created', $fiveDaysAgo, $today, 'date');
        })
        ->matching('Prescriptions', function ($q) use($hospitalization_id) {
          return $q->where(['Prescriptions.hospitalization_id' => $hospitalization_id ]);
        })
        ->order(['PrescriptionMedications.created' => 'DESC']);

        // query to load date buttons
        $medicationPrescriptionDates = $this->Prescriptions->PrescriptionMedications->find()
          ->where(function($exp) use($fiveDaysAgo, $today) {
              return $exp->between('PrescriptionMedications.created', $fiveDaysAgo, $today, 'date');
          })
          ->select([
              'prescription_date' => 'DATE(PrescriptionMedications.created)',
          ])
          ->contain(['Prescriptions'])
          ->matching('Prescriptions', function ($q) use($hospitalization_id) {
            return $q->where(['Prescriptions.hospitalization_id' => $hospitalization_id ]);
          })
          ->group(['prescription_date'])
          ->order(['prescription_date' => 'desc']);


        if ($this->request->is('post')) {
            $preSelectedItensIds = $this->request->data;

            if (!empty($preSelectedItensIds)) {
                $this->request->session()->write('preSelectedItensIds', $preSelectedItensIds);
                return $this->redirect(['action' => 'add']);
            }
        }

        $this->set(compact('prescriptionMedications', 'medicationPrescriptionDates'));
        $this->set('_serialize', ['prescription']);
    }


    public function prescriptionsHistory()
    {
        $session = $this->request->session();
        $hospitalization_id = $session->read('selected-hospitalization');
        $screening_id = $this->request->session()->read('screening_id');
        $current_state = $this->request->session()->read('current-state');
        $lastPrescriptionId = 0;

        $hospitalization = $this->Prescriptions->Hospitalizations->get($hospitalization_id);

        $query = $this->Prescriptions->find('all')
            ->contain(['Professionals.Users'])
            ->where(['Prescriptions.is_default_prescription' => false]);

        $prescriptions = $this->paginate($query);

        $this->set(compact('prescriptions', 'lastPrescriptionId', 'screening_id', 'hospitalization'));
        $this->set('_serialize', ['prescription']);
    }

    public function defaultPrescriptions($prescription_id = null)
    {


        // get all defaultPrescriptions and show them due to user selection in the dropdown

        $session = $this->request->session();
        $hospitalization_id = $session->read('selected-hospitalization');
        $screening_id = $this->request->session()->read('screening_id');
        $current_state = $this->request->session()->read('current-state');
        $lastPrescriptionId = 0;
        $prescriptions = [];
        $associated = ['PrescriptionMedications'];

        $default_prescriptions = $this->Prescriptions->find('all')
         ->contain($associated)
         ->where(['is_default_prescription' => true])
         ->hydrate(['false'])
         ->order(['id'=>'desc']);

        $this->set(compact('default_prescriptions'));
    }

    public function includeDefaultPrescriptions($prescription_id = null)
    {

        // get all defaultPrescriptions and show them due to user selection in the dropdown

        $session = $this->request->session();
        $hospitalization_id = $session->read('selected-hospitalization');
        $screening_id = $this->request->session()->read('screening_id');
        $current_state = $this->request->session()->read('current-state');
        $lastPrescriptionId = 0;
        $prescriptions = [];
        $associated = ['PrescriptionMedications.SystemMedications'];

        $default_prescriptions = $this->Prescriptions->find('all')
         ->contain($associated)
         ->where(['is_default_prescription' => true])
         ->hydrate(['false'])
         ->order(['id'=>'desc']);


        if ($prescription_id != null) {
            $prescription = $this->Prescriptions->get($prescription_id, [
             'contain' => $associated
           ]);
            $this->set(compact('prescription'));
        } else {
            $prescription = $this->Prescriptions->find('all')
            ->contain($associated)
            ->where(['is_default_prescription' => true])
            ->order(['id'=>'desc'])
            ->first();
            $this->set(compact('prescription'));
        }

        if ($this->request->is('post')) {
            $preSelectedItensIds = $this->request->data;

            if (!empty($preSelectedItensIds)) {
                $this->request->session()->write('preSelectedItensIds', $preSelectedItensIds);
                return $this->redirect(['action' => 'add']);
            }
        }

        $this->set(compact('default_prescriptions'));
    }

    public function copyMedicationDataByIds($selectedItensIds = null)
    {
        $copiedMedications = [];

        $excludedFields = [
         'id',
         'created',
         'modified'
      ];

        foreach ($selectedItensIds as $key => $id) {
            if ($id != 0 && !empty($id)) {
                $medicationToCopy =  $this->Prescriptions->PrescriptionMedications->get($id, [
                  'contain' => 'SystemMedications'
                ])
                ->toArray();
                $medicationCopy   = [];

                foreach ($medicationToCopy as $key => $value) {
                    if (!in_array($key, $excludedFields)) {
                        $medicationCopy[$key] = $value;
                    }
                }

                if (!empty($medicationCopy)) {
                    array_push($copiedMedications, $medicationCopy);
                }
            }
        }

        return $copiedMedications;
    }

    /**
     * View method
     *
     * @param string|null $id Prescription id.
     * @return \Cake\Network\Response|null
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $prescription = $this->Prescriptions->get($id, [
            'contain' => [
               'PrescriptionMedications'=> function ($q) {
                   return $q
                       ->contain(['SystemMedications'])
                       ->where(['PrescriptionMedications.is_active' => true]);
               },
               'Professionals.Users'
            ]
        ]);

        $patient_table = TableRegistry::get('Patients');
        $hospitalizations_table = TableRegistry::get('Hospitalizations');

        $patient = $patient_table->get($this->request->session()->read('selected-pacient'));

        $hospitalization = $hospitalizations_table->get($this->request->session()->read('selected-hospitalization'),[
           'contain'=>['HistoricPatientMovements.Rooms', 'Sectors']
        ]);


        $printParam = '';
        $this->set(compact('prescription','printParam','patient','hospitalization'));
        $this->set('_serialize', ['prescription']);
    }

    public function viewDefaultPrescription($id = null) {
      $prescription = $this->Prescriptions->get($id, [
          'contain' => [
             'AvaiableSectors',
             'PrescriptionMedications'=> function ($q) {
                 return $q
                    ->contain(['SystemMedications'])
                    ->where(['PrescriptionMedications.is_active' => true]);
             },
             'Professionals.Users'
          ]
      ]);

      $patient_table = TableRegistry::get('Patients');
      $hospitalizations_table = TableRegistry::get('Hospitalizations');

      $patient = $patient_table->get($this->request->session()->read('selected-pacient'));

      $hospitalization = $hospitalizations_table->get($this->request->session()->read('selected-hospitalization'),[
         'contain'=>['HistoricPatientMovements.Rooms', 'Sectors']
      ]);


      $printParam = '';
      $this->set(compact('prescription','printParam','patient','hospitalization'));
      $this->set('_serialize', ['prescription']);
    }

    public function printPrescription($id = null)
    {
        $prescription = $this->Prescriptions->get($id, [
            'contain' => [
               'PrescriptionMedications'=> function ($q) {
                   return $q
                      ->contain(['SystemMedications'])
                      ->where(['PrescriptionMedications.is_active' => true])
                      ->order(['PrescriptionMedications.print_order']);
               },
               'Professionals.Users'
            ]
        ]);

        $patient_table           = TableRegistry::get('Patients');
        $hospitalizations_table  = TableRegistry::get('Hospitalizations');
        $unitsTable              = TableRegistry::get('Units');

        $patient = $patient_table->get($this->request->session()->read('selected-pacient'));
        $unit    = $unitsTable->get($this->request->session()->read('selected-unit'));

        $hospitalization = $hospitalizations_table->get($this->request->session()->read('selected-hospitalization'),[
           'contain'=>['HistoricPatientMovements.Rooms', 'Sectors']
        ]);

        $this->set(compact('prescription', 'patient', 'hospitalization', 'unit'));
        $this->set('_serialize', ['prescription']);
    }

    /**
     * Add method
     *
     * @return \Cake\Network\Response|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $session                = $this->request->session();

        $prescription           = $this->Prescriptions->newEntity();
        $numberOfMedications    = 0;

        $preSelectedItensIds    = $session->read('preSelectedItensIds');
        $hospitalizationId      = $session->read('selected-hospitalization');
        $currentUserId          = $session->read('Auth.User.id');
        $unitId                 = $session->read('selected-unit');

        $culturesTable              = TableRegistry::get('Cultures');
        $professionalsTable         = TableRegistry::get('Professionals');
        $systemMedicationsTable     = TableRegistry::get('SystemMedications');




        if ($preSelectedItensIds) {
            $preSelectedMedications = $this->copyMedicationDataByIds($preSelectedItensIds);
            $numberOfMedications  = count($preSelectedMedications);
            $this->set(compact('preSelectedMedications'));
        }

        $professional       = $professionalsTable->find('all')->where(['user_id' => $currentUserId])->first();
        $currentSector      = $this->Prescriptions->Hospitalizations->getCurrentSector($hospitalizationId);
        $culturesWithGerms  = $culturesTable->find('list', [
            'fields' => ['id', 'germe'],
            'conditions' => ["Cultures.hospitalization_id" => $hospitalizationId, 'Cultures.resultado' => 1],
        ]);

        $systemMedications = $systemMedicationsTable->find('all')
          ->select(['id', 'name', 'is_active'])
          ->where(["SystemMedications.unit_id" => $unitId])
          ->hydrate(false);

          $systemMedicationsOpt = [];
          foreach ($systemMedications as $key => $sysMed) {
            array_push(
              $systemMedicationsOpt,
              [
                'text' => $sysMed['name'],
                'value' => ($sysMed['is_active']) ? $sysMed['id'] : '',
                'disabled' => !$sysMed['is_active']
              ]
            );
          }

        $associated = ['PrescriptionMedications.Cultures'];

        if(!$professional) {
          $this->Flash->error(__('É necessário estar cadastrado como profissional para realizar esta ação.'));
          return $this->redirect($this->referer());
        }

        if ($this->request->is('post')) {

            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data, ['associated' => $associated]);

            if ($this->Prescriptions->save($prescription, ['associated' => $associated])) {
                $session->delete('preSelectedItensIds');
                $this->Flash->success(__('O registro  foi salvo'));
                return $this->redirect(['action' => 'prescriptionsHistory']);
            } else {
                $this->Flash->error(__('O registro não foi salvo'));
                return $this->redirect(['action' => 'prescriptionsHistory']);
            }
        }

        $this->set(compact('prescription', 'currentSector', 'numberOfMedications', 'culturesWithGerms','professional', 'systemMedicationsOpt', 'hospitalizationId','systemMedications'));
        $this->set('_serialize', ['prescription']);
    }

    public function getMedicationsSuggestions()
    {
        $this->autoRender = false; // avoid to render view

        $system_medications_table = TableRegistry::get('SystemMedications');
        $medications = $system_medications_table->find('list', [
           'keyField' => 'id',
           'valueField' => 'name'
        ]);

        $res = [];
        foreach ($medications as $key => $value) {
            array_push($res, [ "label" => $value, "value" => $key ]);
        }

        $res = json_encode($res);

        $this->response->type('json');
        $this->response->body($res);

        return $this->response;
    }

    public function getSelectedMedication($id=null){
      $this->autoRender = false; // avoid to render view

     $system_medications_table = TableRegistry::get('SystemMedications');
     $medication = $system_medications_table->get($id)->toArray();

     $res = json_encode($medication);

     $this->response->type('json');
     $this->response->body($res);

     return $this->response;
   }



    public function addDefaultPrescription()
    {
        $prescription           = $this->Prescriptions->newEntity();
        $session                = $this->request->session();

        $professionalsTable     = TableRegistry::get('Professionals');
        $sectorsTable           = TableRegistry::get('Sectors');
        $systemMedicationsTable = TableRegistry::get('SystemMedications');

        $currentUserId          = $session->read('Auth.User.id');
        $hospitalizationId      = $session->read('selected-hospitalization');
        $unitId                 = $session->read('selected-unit');

        $currentSector = $this->Prescriptions->Hospitalizations->getCurrentSector($hospitalizationId);

        $sectors = $sectorsTable->find('list', [
            'keyField' => 'id',
            'valueField' => 'name'
         ])
         ->where(["Sectors.unit_id" => $unitId]);

         $systemMedications = $systemMedicationsTable->find('all')
           ->select(['id', 'name', 'is_active'])
           ->where(["SystemMedications.unit_id" => $unitId])
           ->hydrate(false);

        $professional = $professionalsTable->find('all')
          ->where(['user_id' => $currentUserId])
          ->first();

        if(!$professional) {
          $this->Flash->error(__('É necessário estar cadastrado como profissional para realizar esta ação.'));
          return $this->redirect($this->referer());
        }

        $associated = [
            'PrescriptionMedications', 'Sectors', 'Professionals','AvaiableSectors'
        ];

        if ($this->request->is('post')) {
            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data, [
               'associated' => $associated
            ]);

            $prescription->is_default_prescription  = true;

            if ($this->Prescriptions->save($prescription)) {
                $this->Flash->success(__('O registro  foi salvo'));
                return $this->redirect(['action' => 'defaultPrescriptions']);
            } else {
                $this->Flash->error(__('O registro não foi salvo'));
                return $this->redirect(['action' => 'defaultPrescriptions']);
            }
        }
        $numberOfMedications = (!empty($prescription['prescription_medications'])) ? count($prescription['prescription_medications']) : 0;
        $this->set(compact('prescription','currentSector', 'sectors', 'professional', 'systemMedications', 'numberOfMedications'));
        $this->set('_serialize', ['prescription']);
    }

    public function editDefaultPrescription($id) {

      $prescription = $this->Prescriptions->get($id, [
          'contain' => [
            'AvaiableSectors',
             'PrescriptionMedications'=> function ($q) {
                 return $q
                     ->contain(['SystemMedications'])
                     ->where(['PrescriptionMedications.is_active' => true]);
             },
             'Professionals.Users'
          ]
      ]);

      $session                = $this->request->session();
      $sectorsTable           = TableRegistry::get('Sectors');
      $professionalsTable     = TableRegistry::get('Professionals');
      $systemMedicationsTable = TableRegistry::get('SystemMedications');
      $unitId                 = $session->read('selected-unit');
      $currentUserId          = $session->read('Auth.User.id');

      $professional = $professionalsTable->find('all')
        ->where(['user_id' => $currentUserId])
        ->first();

      if(!$professional) {
        $this->Flash->error(__('É necessário estar cadastrado como profissional para realizar esta ação.'));
        return $this->redirect($this->referer());
      }

      $sectors = $sectorsTable->find('list', [
          'keyField' => 'id',
          'valueField' => 'name'
       ])
       ->where(["Sectors.unit_id" => $unitId]);

       $systemMedications = $systemMedicationsTable->find('all')
         ->select(['id', 'name', 'is_active'])
         ->where(["SystemMedications.unit_id" => $unitId])
         ->hydrate(false);

       $systemMedicationsOpt = [];
       foreach ($systemMedications as $key => $sysMed) {
         array_push(
           $systemMedicationsOpt,
           [
             'text' => $sysMed['name'],
             'value' => ($sysMed['is_active']) ? $sysMed['id'] : '',
             'disabled' => !$sysMed['is_active']
           ]
         );
       }

      $associated = [
          'PrescriptionMedications', 'Sectors', 'Professionals', 'AvaiableSectors'
      ];

      if ($this->request->is(['post','put'])) {
          $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data, [
             'associated' => $associated
          ]);

          $prescription->is_default_prescription  = true;

          if ($this->Prescriptions->save($prescription)) {
              $this->Flash->success(__('O registro foi salvo'));
              return $this->redirect(['action' => 'defaultPrescriptions']);
          } else {
              $this->Flash->error(__('O registro não foi salvo'));
              return $this->redirect(['action' => 'defaultPrescriptions']);
          }
      }

      $numberOfMedications = (!empty($prescription['prescription_medications'])) ? count($prescription['prescription_medications']) : 0;

      $this->set(compact('prescription', 'sectors','numberOfMedications','systemMedications','systemMedicationsOpt'));

      $this->set('_serialize', ['prescription']);
    }

    /**
     * Edit method
     *
     * @param string|null $id Prescription id.
     * @return \Cake\Network\Response|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Network\Exception\NotFoundException When record not found.
     */
    public function edit($prescription_id = null)
    {
        $cultures_table = TableRegistry::get('Cultures');
        $session = $this->request->session();
        $unit_id = $session->read('selected-unit');
        $hospitalization_id = $session->read('selected-hospitalization');

        $cultures_with_germs = $cultures_table->find('list')
            ->where(["Cultures.hospitalization_id" => $hospitalization_id, 'Cultures.resultado' => 1])
        ;

        $i = 0;
        $cultures_ids = [];

        foreach ($cultures_with_germs as $key => $culture) {
            $cultures_ids[$i] = $culture['id'];
            $i++;
        }

        $prescription = $this->Prescriptions->get($prescription_id, [
            'contain' => ['Medications.Cultures'],
        ]);

        // inicio filtrando só as medicações ativas na prescrição
        $active_medications = [];
        $i = 0;

        if (isset($prescription->medications)) {
            foreach ($prescription->medications as $key => $medication) {
                if ($medication['is_active']) {
                    $active_medications[$i] = $medication;
                }

                $i++;
            }
        }

        $prescription->medications = $active_medications;

        // fim filtrando só as medicações ativas na prescrição

        if ($this->request->is(['patch', 'post', 'put'])) {
            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data);

            if ($this->Prescriptions->save($prescription)) {
                $this->Flash->success(__('O registro  foi salvo'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('O registro não foi salvo'));
            }
        }

        $signs = $this->Prescriptions->Medications->Signs
            ->find('all', ['limit' => 200])
            ->where(["Signs.unit_id" => $unit_id]);
        $this->set(compact('prescription', 'prescription_id', 'cultures_with_germs', 'cultures_ids', 'active_medications'));
        $this->set('_serialize', ['prescription']);
    }

    public function printView($prescription_id = null)
    {
        $session = $this->request->session();

        $patientTable = TableRegistry::get('Patients');
        $hospitalizations_table = TableRegistry::get('Hospitalizations');
        $cultures_table = TableRegistry::get('Cultures');
        $units_table = TableRegistry::get('Units');

        $unit_id = $session->read('selected-unit');
        $pacient_id = $session->read('selected-pacient');
        $hospitalization_id = $session->read('selected-hospitalization');
        $unit_id = $session->read('selected-unit');
        $is_print_view = true;

        $hospitalization = $hospitalizations_table->get($hospitalization_id, ['contain' => 'Discharges']);
        $unit = $units_table->get($unit_id);
        $patient = $patientTable->get($pacient_id);

        $cultures_with_germs = $cultures_table->find('list', [
            'fields' => ['germe', 'tipo_resistencia'],
            'conditions' => ["Cultures.hospitalization_id" => $hospitalization_id, 'Cultures.resultado' => 1],
        ])->toArray();

        //monatndo uma escala niveis de resistencia
        $niveis_resistencia = [
            'Sem resistência' => 0,
            'Resistente' => 1,
            'Multiresistente' => 2,
            'Panresistente' => 3,
        ];

        // montando um array das resistências de todos os germes e atribuindo os valores numéricos dos níveis
        $resistesncias = [];
        foreach ($cultures_with_germs as $key => $resistencia) {
            if ($resistencia) {
                $resistesncias[$resistencia] = $niveis_resistencia[$resistencia];
            }
        }
        //pegando a resistencia de nível mais alto
        $top_resitence = '';
        if (count($resistesncias) > 0) {
            $top_resitence = array_search(max($resistesncias), $resistesncias);
        }

        $prescription = $this->Prescriptions->get($prescription_id, [
            'contain' => ['Medications.Cultures'],
        ]);

        // inicio filtrando só as medicações ativas na prescrição
        $active_medications = [];
        $i = 0;

        if (isset($prescription->medications)) {
            foreach ($prescription->medications as $key => $medication) {
                if ($medication['is_active']) {
                    $active_medications[$i] = $medication;
                }

                $i++;
            }
        }

        $prescription->medications = $active_medications;

        // fim filtrando só as medicações ativas na prescrição

        if ($this->request->is(['patch', 'post', 'put'])) {
            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data);

            if ($this->Prescriptions->save($prescription)) {
                $this->Flash->success(__('O registro  foi salvo'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('O registro não foi salvo'));
            }
        }

        $signs = $this->Prescriptions->Medications->Signs
            ->find('all', ['limit' => 200])
            ->where(["Signs.unit_id" => $unit_id]);
        $this->set(compact('top_resitence', 'hospitalization', 'patient', 'is_print_view', 'unit', 'prescription', 'prescription_id', 'cultures_with_germs', 'active_medications'));
        $this->set('_serialize', ['prescription']);
    }

    public function printRecipes($prescription_id = null)
    {
        $session = $this->request->session();

        $patientTable = TableRegistry::get('Patients');
        $hospitalizations_table = TableRegistry::get('Hospitalizations');
        $cultures_table = TableRegistry::get('Cultures');
        $units_table = TableRegistry::get('Units');

        $unit_id = $session->read('selected-unit');
        $pacient_id = $session->read('selected-pacient');
        $hospitalization_id = $session->read('selected-hospitalization');
        $unit_id = $session->read('selected-unit');
        $is_print_view = true;

        $hospitalization = $hospitalizations_table->get($hospitalization_id, ['contain' => 'Discharges']);
        $unit = $units_table->get($unit_id);
        $patient = $patientTable->get($pacient_id);

        $cultures_with_germs = $cultures_table->find('list', [
            'fields' => ['germe', 'tipo_resistencia'],
            'conditions' => ["Cultures.hospitalization_id" => $hospitalization_id, 'Cultures.resultado' => 1],
        ])->toArray();

        // echo"<pre>";
        //   print_r($cultures_with_germs);
        // echo"</pre>";

        //monatndo uma escala niveis de resistencia
        $niveis_resistencia = [
            'Sem resistência' => 0,
            'Resistente' => 1,
            'Multiresistente' => 2,
            'Panresistente' => 3,
        ];

        // montando um array das resistências de todos os germes e atribuindo os valores numéricos dos níveis
        $resistesncias = [];
        foreach ($cultures_with_germs as $key => $resistencia) {
            if ($resistencia) {
                $resistesncias[$resistencia] = $niveis_resistencia[$resistencia];
            }
        }
        //pegando a resistencia de nível mais alto
        $top_resitence = '';
        if (count($resistesncias) > 0) {
            $top_resitence = array_search(max($resistesncias), $resistesncias);
        }

        $prescription = $this->Prescriptions->get($prescription_id, [
            'contain' => ['Medications.Cultures'],
        ]);

        // inicio filtrando só as medicações ativas na prescrição
        $active_medications = [];
        $i = 0;

        if (isset($prescription->medications)) {
            foreach ($prescription->medications as $key => $medication) {
                if ($medication['is_active']) {
                    $active_medications[$i] = $medication;
                }

                $i++;
            }
        }

        $prescription->medications = $active_medications;

        // fim filtrando só as medicações ativas na prescrição

        if ($this->request->is(['patch', 'post', 'put'])) {
            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data);

            if ($this->Prescriptions->save($prescription)) {
                $this->Flash->success(__('O registro  foi salvo'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('O registro não foi salvo'));
            }
        }

        $signs = $this->Prescriptions->Medications->Signs
            ->find('all', ['limit' => 200])
            ->where(["Signs.unit_id" => $unit_id]);
        $this->set(compact('top_resitence', 'hospitalization', 'patient', 'is_print_view', 'unit', 'prescription', 'prescription_id', 'cultures_with_germs', 'active_medications'));
        $this->set('_serialize', ['prescription']);
    }

    public function printJustification($prescription_id = null)
    {
        $session = $this->request->session();

        $patientTable = TableRegistry::get('Patients');
        $hospitalizations_table = TableRegistry::get('Hospitalizations');
        $cultures_table = TableRegistry::get('Cultures');
        $units_table = TableRegistry::get('Units');

        $unit_id = $session->read('selected-unit');
        $pacient_id = $session->read('selected-pacient');
        $hospitalization_id = $session->read('selected-hospitalization');
        $unit_id = $session->read('selected-unit');
        $is_print_view = true;

        $hospitalization = $hospitalizations_table->get($hospitalization_id, ['contain' => 'Discharges']);
        $unit = $units_table->get($unit_id);
        $patient = $patientTable->get($pacient_id);

        $cultures_with_germs = $cultures_table->find('list', [
            'fields' => ['germe', 'tipo_resistencia'],
            'conditions' => ["Cultures.hospitalization_id" => $hospitalization_id, 'Cultures.resultado' => 1],
        ])->toArray();

        // echo"<pre>";
        //   print_r($cultures_with_germs);
        // echo"</pre>";

        //monatndo uma escala niveis de resistencia
        $niveis_resistencia = [
            'Sem resistência' => 0,
            'Resistente' => 1,
            'Multiresistente' => 2,
            'Panresistente' => 3,
        ];

        // montando um array das resistências de todos os germes e atribuindo os valores numéricos dos níveis
        $resistesncias = [];
        foreach ($cultures_with_germs as $key => $resistencia) {
            if ($resistencia) {
                $resistesncias[$resistencia] = $niveis_resistencia[$resistencia];
            }
        }
        //pegando a resistencia de nível mais alto
        $top_resitence = '';
        if (count($resistesncias) > 0) {
            $top_resitence = array_search(max($resistesncias), $resistesncias);
        }

        $prescription = $this->Prescriptions->get($prescription_id, [
            'contain' => ['Medications.Cultures'],
        ]);

        // inicio filtrando só as medicações ativas na prescrição
        $active_medications = [];
        $i = 0;

        if (isset($prescription->medications)) {
            foreach ($prescription->medications as $key => $medication) {
                if ($medication['is_active']) {
                    $active_medications[$i] = $medication;
                }

                $i++;
            }
        }

        $prescription->medications = $active_medications;

        // fim filtrando só as medicações ativas na prescrição

        if ($this->request->is(['patch', 'post', 'put'])) {
            $prescription = $this->Prescriptions->patchEntity($prescription, $this->request->data);

            if ($this->Prescriptions->save($prescription)) {
                $this->Flash->success(__('O registro  foi salvo'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('O registro não foi salvo'));
            }
        }

        $signs = $this->Prescriptions->Medications->Signs
            ->find('all', ['limit' => 200])
            ->where(["Signs.unit_id" => $unit_id]);
        $this->set(compact('top_resitence', 'hospitalization', 'patient', 'is_print_view', 'unit', 'prescription', 'prescription_id', 'cultures_with_germs', 'active_medications'));
        $this->set('_serialize', ['prescription']);
    }

    /**
     * Delete method
     *
     * @param string|null $id Prescription id.
     * @return \Cake\Network\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */

    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);

        $prescription = $this->Prescriptions->get($id);

        //somente desativar a Prescription e não deletar
        $prescription['is_active'] = 0;

        if ($this->Prescriptions->save($prescription)) {
            $this->Flash->success(__('O registro foi deletado'));
        } else {
            $this->Flash->error(__('O registro não foi deletado'));
        }

        return $this->redirect(['action' => 'index', $prescription_id]);
    }

    public function deactivatePrescriptionMedication($id = null)
    {
        $res = [
          'status' => false,
          'message' => 'Não foi possível desabilitar o medicamento'
        ];

        $this->request->allowMethod(['post', 'delete','ajax']);
        $prescriptionMedication = $this->Prescriptions->PrescriptionMedications->get($id);
        $prescriptionMedication->is_active = false;
        if ($this->Prescriptions->PrescriptionMedications->save($prescriptionMedication)) {
          $res = [
            'status' => true,
            'message' => 'Medicamento desabilitado com sucesso.'
          ];
        }

        $res = json_encode($res);
        $this->response->type('json');
        $this->response->body($res);

        return $this->response;

    }

    public function validatePrescription($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);

        $prescription = $this->Prescriptions->get($id);

        //somente desativar a Prescription e não deletar
        $prescription['validated'] = 1;
        $prescription['validation_date'] = date('Y-m-d H:i:s');

        if ($this->Prescriptions->save($prescription)) {
            $this->Flash->success(__('O registro foi validado'));
        } else {
            $this->Flash->error(__('O registro não foi validado'));
        }

        return $this->redirect(['action' => 'pharmaceutical-validation']);
    }

    public function invalidatePrescription($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);

        $prescription = $this->Prescriptions->get($id);

        //somente desativar a Prescription e não deletar
        $prescription['validated'] = 0;

        if ($this->Prescriptions->save($prescription)) {
            $this->Flash->success(__('O registro foi invalidado'));
        } else {
            $this->Flash->error(__('O registro não foi invalidado'));
        }

        return $this->redirect(['action' => 'pharmaceutical-validation']);
    }

    private function dateToDatabase($d) {
      $d = str_replace('/', '-', $d);
      return date('Y-m-d', strtotime($d));
    }
}
